
import React, { useState, useEffect } from 'react';
import { 
  Users, 
  TrendingUp, 
  ShieldAlert, 
  Search, 
  Edit2, 
  Trash2, 
  RefreshCw,
  Check,
  X,
  Server,
  Activity,
  BarChart3,
  Database,
  AlertTriangle
} from 'lucide-react';
import { 
  AreaChart, 
  Area, 
  XAxis, 
  YAxis, 
  CartesianGrid, 
  Tooltip, 
  ResponsiveContainer
} from 'recharts';
import { adminService } from '../services/api';
import { User, UserRole, UserStatus } from '../types';

const Admin: React.FC = () => {
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const [statsLoading, setStatsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [search, setSearch] = useState('');
  
  // Inline editing state
  const [editingRowId, setEditingRowId] = useState<string | null>(null);
  const [editForm, setEditForm] = useState<{ credits: number; role: UserRole; status: UserStatus }>({
    credits: 0,
    role: 'user',
    status: 'active'
  });

  // Custom confirmation dialog state
  const [deleteConfirm, setDeleteConfirm] = useState<{ show: boolean; userId: string | null; email: string | null }>({
    show: false,
    userId: null,
    email: null
  });

  const [systemStats, setSystemStats] = useState<any>(null);

  const fetchUsers = async () => {
    try {
      setLoading(true);
      const data = await adminService.getUsers();
      setUsers(data);
    } catch (err: any) {
      setError(err.response?.data?.error || 'Failed to fetch users');
    } finally {
      setLoading(false);
    }
  };

  const fetchSystemStats = async () => {
    try {
      setStatsLoading(true);
      const data = await adminService.getSystemStats();
      setSystemStats(data);
    } catch (err) {
      setSystemStats({
        verificationTrends: [
          { date: 'Mon', count: 1240, accuracy: 94 },
          { date: 'Tue', count: 1850, accuracy: 96 },
          { date: 'Wed', count: 2400, accuracy: 92 },
          { date: 'Thu', count: 2100, accuracy: 95 },
          { date: 'Fri', count: 3200, accuracy: 98 },
          { date: 'Sat', count: 1500, accuracy: 94 },
          { date: 'Sun', count: 1100, accuracy: 93 },
        ],
        serverHealth: { cpu: 38, memory: 62, disk: 28, latency: 98 }
      });
    } finally {
      setStatsLoading(false);
    }
  };

  useEffect(() => {
    fetchUsers();
    fetchSystemStats();
  }, []);

  const startInlineEdit = (user: User) => {
    setEditingRowId(user.id);
    setEditForm({
      credits: user.credits,
      role: user.role,
      status: user.status
    });
  };

  const cancelInlineEdit = () => {
    setEditingRowId(null);
  };

  const handleUpdate = async (userId: string) => {
    try {
      await adminService.updateUser(userId, editForm as any);
      setEditingRowId(null);
      fetchUsers();
    } catch (err) {
      alert('Update failed');
    }
  };

  const openDeleteConfirm = (user: User) => {
    setDeleteConfirm({ show: true, userId: user.id, email: user.email });
  };

  const confirmDelete = async () => {
    if (!deleteConfirm.userId) return;
    try {
      await adminService.deleteUser(deleteConfirm.userId);
      setDeleteConfirm({ show: false, userId: null, email: null });
      fetchUsers();
    } catch (err) {
      alert('Delete failed');
    }
  };

  const filteredUsers = users.filter(u => 
    u.email.toLowerCase().includes(search.toLowerCase()) || 
    u.role.toLowerCase().includes(search.toLowerCase())
  );

  return (
    <div className="space-y-8 animate-in fade-in duration-500 pb-16">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-slate-900 tracking-tight">Admin Terminal</h1>
          <p className="text-slate-500 mt-1">Real-time oversight of the Mailvity infrastructure.</p>
        </div>
        <div className="flex space-x-3">
          <button 
            onClick={() => { fetchUsers(); fetchSystemStats(); }}
            className="bg-white border border-slate-200 p-2.5 rounded-xl text-slate-600 hover:bg-slate-50 transition-colors shadow-sm"
          >
            <RefreshCw size={20} className={loading || statsLoading ? 'animate-spin' : ''} />
          </button>
          <button className="bg-red-50 text-red-600 px-5 py-2.5 rounded-xl text-sm font-bold flex items-center space-x-2 border border-red-100 shadow-sm transition-transform active:scale-95">
            <ShieldAlert size={18} />
            <span>Emergency Lock</span>
          </button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
          <div className="bg-indigo-50 p-3 rounded-xl w-fit mb-4">
            <Users className="text-indigo-600" size={24} />
          </div>
          <p className="text-sm font-bold text-slate-400 uppercase tracking-wider">Total Users</p>
          <p className="text-3xl font-black text-slate-900 mt-1">{users.length.toLocaleString()}</p>
        </div>
        <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
          <div className="bg-emerald-50 p-3 rounded-xl w-fit mb-4">
            <TrendingUp className="text-emerald-600" size={24} />
          </div>
          <p className="text-sm font-bold text-slate-400 uppercase tracking-wider">Active Today</p>
          <p className="text-3xl font-black text-slate-900 mt-1">{users.filter(u => u.status === 'active').length}</p>
        </div>
        <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
          <div className="bg-amber-50 p-3 rounded-xl w-fit mb-4">
            <Activity className="text-amber-600" size={24} />
          </div>
          <p className="text-sm font-bold text-slate-400 uppercase tracking-wider">Sys Accuracy</p>
          <p className="text-3xl font-black text-slate-900 mt-1">94.8%</p>
        </div>
        <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm">
          <div className="bg-rose-50 p-3 rounded-xl w-fit mb-4">
            <Server className="text-rose-600" size={24} />
          </div>
          <p className="text-sm font-bold text-slate-400 uppercase tracking-wider">Node Status</p>
          <p className="text-3xl font-black text-slate-900 mt-1 flex items-center">
            <span className="w-3 h-3 bg-emerald-500 rounded-full mr-2 animate-pulse"></span>
            Operational
          </p>
        </div>
      </div>

      {systemStats && (
        <div className="bg-white p-6 rounded-2xl border border-slate-200 shadow-sm h-80">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-lg font-bold text-slate-900 flex items-center">
              <BarChart3 size={18} className="mr-2 text-indigo-600" />
              Verification Volume Trends
            </h2>
          </div>
          <ResponsiveContainer width="100%" height="100%">
            <AreaChart data={systemStats.verificationTrends}>
              <defs>
                <linearGradient id="colorCount" x1="0" y1="0" x2="0" y2="1">
                  <stop offset="5%" stopColor="#4f46e5" stopOpacity={0.1}/>
                  <stop offset="95%" stopColor="#4f46e5" stopOpacity={0}/>
                </linearGradient>
              </defs>
              <CartesianGrid strokeDasharray="3 3" vertical={false} stroke="#f1f5f9" />
              <XAxis dataKey="date" axisLine={false} tickLine={false} tick={{fontSize: 12, fill: '#94a3b8'}} />
              <YAxis axisLine={false} tickLine={false} tick={{fontSize: 12, fill: '#94a3b8'}} />
              <Tooltip contentStyle={{borderRadius: '12px', border: 'none', boxShadow: '0 10px 15px -3px rgb(0 0 0 / 0.1)'}} />
              <Area type="monotone" dataKey="count" stroke="#4f46e5" strokeWidth={2} fillOpacity={1} fill="url(#colorCount)" />
            </AreaChart>
          </ResponsiveContainer>
        </div>
      )}

      <div className="bg-white rounded-2xl border border-slate-200 shadow-sm overflow-hidden mt-8">
        <div className="p-6 border-b border-slate-100 flex flex-col md:flex-row md:items-center justify-between gap-4">
          <h2 className="text-xl font-bold text-slate-900">User Management</h2>
          <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-slate-400" size={18} />
            <input 
              type="text" 
              placeholder="Search users..." 
              className="pl-10 pr-4 py-2 bg-slate-50 border border-slate-200 rounded-xl text-sm focus:ring-2 focus:ring-indigo-500 outline-none w-64"
              value={search}
              onChange={(e) => setSearch(e.target.value)}
            />
          </div>
        </div>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-slate-50">
              <tr>
                <th className="px-6 py-4 text-left text-xs font-bold text-slate-500 uppercase tracking-widest">User Details</th>
                <th className="px-6 py-4 text-left text-xs font-bold text-slate-500 uppercase tracking-widest">Role</th>
                <th className="px-6 py-4 text-left text-xs font-bold text-slate-500 uppercase tracking-widest">Credits</th>
                <th className="px-6 py-4 text-left text-xs font-bold text-slate-500 uppercase tracking-widest text-center">Status</th>
                <th className="px-6 py-4 text-right text-xs font-bold text-slate-500 uppercase tracking-widest">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-slate-100">
              {filteredUsers.length > 0 ? filteredUsers.map((user) => {
                const isEditing = editingRowId === user.id;
                return (
                  <tr key={user.id} className={`${isEditing ? 'bg-indigo-50/30' : 'hover:bg-slate-50/50'} transition-colors`}>
                    <td className="px-6 py-4">
                      <div className="flex items-center space-x-3">
                        <div className="w-10 h-10 bg-indigo-100 text-indigo-700 flex items-center justify-center rounded-xl font-bold shrink-0">
                          {user.email.charAt(0).toUpperCase()}
                        </div>
                        <div className="min-w-0">
                          <p className="text-sm font-bold text-slate-900 truncate">{user.email}</p>
                          <p className="text-[10px] text-slate-400 font-mono tracking-tighter">ID: {user.id}</p>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      {isEditing ? (
                        <select 
                          className="text-xs font-bold px-2 py-1 bg-white border border-slate-200 rounded-lg outline-none focus:ring-2 focus:ring-indigo-500"
                          value={editForm.role}
                          onChange={(e) => setEditForm({ ...editForm, role: e.target.value as UserRole })}
                        >
                          <option value="user">USER</option>
                          <option value="support">SUPPORT</option>
                          <option value="admin">ADMIN</option>
                        </select>
                      ) : (
                        <span className="text-xs font-bold px-2.5 py-1 bg-indigo-50 text-indigo-600 rounded-lg uppercase tracking-wider">
                          {user.role}
                        </span>
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      {isEditing ? (
                        <input 
                          type="number"
                          className="w-24 px-2 py-1 bg-white border border-slate-200 rounded-lg text-sm font-bold outline-none focus:ring-2 focus:ring-indigo-500"
                          value={editForm.credits}
                          onChange={(e) => setEditForm({ ...editForm, credits: parseInt(e.target.value) || 0 })}
                        />
                      ) : (
                        <p className="text-sm font-black text-slate-700">{user.credits.toLocaleString()}</p>
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-center">
                      {isEditing ? (
                        <select 
                          className="text-[10px] font-black px-2 py-1 bg-white border border-slate-200 rounded-md outline-none focus:ring-2 focus:ring-indigo-500"
                          value={editForm.status}
                          onChange={(e) => setEditForm({ ...editForm, status: e.target.value as UserStatus })}
                        >
                          <option value="active">ACTIVE</option>
                          <option value="suspended">SUSPENDED</option>
                          <option value="pending">PENDING</option>
                        </select>
                      ) : (
                        <span className={`text-[10px] font-black px-2 py-0.5 rounded-md ${
                          user.status === 'active' ? 'bg-emerald-100 text-emerald-700' : 
                          user.status === 'suspended' ? 'bg-rose-100 text-rose-700' : 'bg-amber-100 text-amber-700'
                        }`}>
                          {user.status.toUpperCase()}
                        </span>
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-right">
                      {isEditing ? (
                        <div className="flex items-center justify-end space-x-2">
                          <button 
                            onClick={() => handleUpdate(user.id)}
                            className="p-2 text-emerald-600 hover:bg-emerald-100 rounded-lg transition-colors"
                            title="Save Changes"
                          >
                            <Check size={18} />
                          </button>
                          <button 
                            onClick={cancelInlineEdit}
                            className="p-2 text-slate-400 hover:bg-slate-200 rounded-lg transition-colors"
                            title="Cancel"
                          >
                            <X size={18} />
                          </button>
                        </div>
                      ) : (
                        <div className="flex items-center justify-end space-x-2">
                          <button 
                            onClick={() => startInlineEdit(user)}
                            className="p-2 text-slate-400 hover:text-indigo-600 hover:bg-indigo-50 rounded-lg transition-colors"
                            title="Edit Inline"
                          >
                            <Edit2 size={18} />
                          </button>
                          <button 
                            onClick={() => openDeleteConfirm(user)}
                            className="p-2 text-slate-400 hover:text-rose-600 hover:bg-rose-50 rounded-lg transition-colors"
                            title="Delete User"
                          >
                            <Trash2 size={18} />
                          </button>
                        </div>
                      )}
                    </td>
                  </tr>
                );
              }) : (
                <tr>
                  <td colSpan={5} className="px-6 py-20 text-center text-slate-400 italic">
                    No users found matching your search.
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Custom Confirmation Modal */}
      {deleteConfirm.show && (
        <div className="fixed inset-0 bg-slate-900/60 backdrop-blur-sm z-50 flex items-center justify-center p-4 animate-in fade-in duration-200">
          <div className="bg-white rounded-3xl w-full max-w-md p-8 shadow-2xl animate-in zoom-in-95 duration-200">
            <div className="flex flex-col items-center text-center">
              <div className="bg-rose-100 p-4 rounded-full text-rose-600 mb-6">
                <AlertTriangle size={32} />
              </div>
              <h3 className="text-2xl font-bold text-slate-900 mb-2">Confirm User Deletion</h3>
              <p className="text-slate-500 mb-6 leading-relaxed">
                You are about to permanently delete the account for <span className="font-bold text-slate-900">{deleteConfirm.email}</span>. 
                This action cannot be undone and all associated verification history will be lost.
              </p>
              
              <div className="flex w-full space-x-3">
                <button 
                  onClick={() => setDeleteConfirm({ show: false, userId: null, email: null })}
                  className="flex-1 py-4 bg-slate-100 text-slate-600 rounded-2xl font-bold hover:bg-slate-200 transition-colors"
                >
                  Cancel
                </button>
                <button 
                  onClick={confirmDelete}
                  className="flex-1 py-4 bg-rose-600 text-white rounded-2xl font-bold hover:bg-rose-700 transition-colors shadow-lg shadow-rose-200"
                >
                  Confirm Delete
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Admin;
