
import React, { useState, useEffect, useRef } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { 
  CheckCircle, 
  XCircle, 
  AlertCircle, 
  HelpCircle, 
  Download, 
  RefreshCw,
  Search,
  ChevronRight,
  Shield,
  Zap,
  Globe,
  Mail as MailIcon
} from 'lucide-react';
import { VerificationResult, Job, VerificationStatus } from '../types';
import { verificationService } from '../services/api';

const Results: React.FC = () => {
  const { jobId } = useParams<{ jobId: string }>();
  const navigate = useNavigate();
  const [job, setJob] = useState<Job | null>(null);
  const [loading, setLoading] = useState(true);
  const [progressStage, setProgressStage] = useState(0);
  const [filter, setFilter] = useState<VerificationStatus | 'All'>('All');
  const [search, setSearch] = useState('');
  const pollTimer = useRef<number | null>(null);

  const stages = [
    { label: 'Syntax Validation', icon: <Zap size={16} />, detail: 'Checking RFC compliance and mailbox structure...' },
    { label: 'DNS / MX Records', icon: <Globe size={16} />, detail: 'Querying domain mail servers and reputation...' },
    { label: 'SMTP Handshake', icon: <MailIcon size={16} />, detail: 'Connecting to target server for mailbox verification...' },
    { label: 'AI Confidence Scoring', icon: <Shield size={16} />, detail: 'Analyzing signals and determining deliverability...' }
  ];

  useEffect(() => {
    const fetchResults = async () => {
      if (!jobId) return;

      try {
        const data = await verificationService.getResults(jobId);
        setJob(data);

        // Map backend status to progress stages for the UI
        if (data.status === 'processing') {
          // Progress stages are simulated for UX while polling
          setProgressStage(prev => Math.min(prev + 1, stages.length - 1));
        }

        if (data.status === 'completed' || data.status === 'failed') {
          setLoading(false);
          if (pollTimer.current) clearInterval(pollTimer.current);
        }
      } catch (error) {
        console.error('Failed to fetch results');
      }
    };

    fetchResults();
    pollTimer.current = window.setInterval(fetchResults, 3000);

    return () => {
      if (pollTimer.current) clearInterval(pollTimer.current);
    };
  }, [jobId]);

  const filteredResults = job?.results?.filter(res => {
    const matchesSearch = res.email.toLowerCase().includes(search.toLowerCase());
    const matchesFilter = filter === 'All' || res.status === filter;
    return matchesSearch && matchesFilter;
  });

  const getStatusBadge = (status: VerificationStatus) => {
    switch (status) {
      case VerificationStatus.VALID:
        return <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-bold bg-emerald-100 text-emerald-800">
          <CheckCircle size={12} className="mr-1" /> {status}
        </span>;
      case VerificationStatus.INVALID:
        return <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-bold bg-rose-100 text-rose-800">
          <XCircle size={12} className="mr-1" /> {status}
        </span>;
      case VerificationStatus.CATCH_ALL:
        return <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-bold bg-amber-100 text-amber-800">
          <AlertCircle size={12} className="mr-1" /> {status}
        </span>;
      case VerificationStatus.UNKNOWN:
        return <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-bold bg-slate-100 text-slate-800">
          <HelpCircle size={12} className="mr-1" /> {status}
        </span>;
    }
  };

  const stats = job?.results?.reduce((acc, curr) => {
    acc[curr.status] = (acc[curr.status] || 0) + 1;
    return acc;
  }, {} as Record<string, number>) || {};

  if (loading) {
    return (
      <div className="flex flex-col items-center justify-center min-h-[60vh] max-w-lg mx-auto space-y-12">
        <div className="text-center space-y-3">
          <div className="bg-indigo-600 p-4 rounded-3xl w-fit mx-auto mb-6 shadow-2xl shadow-indigo-100 animate-pulse">
            <RefreshCw className="animate-spin text-white" size={40} />
          </div>
          <h2 className="text-2xl font-bold text-slate-900">Verification in Progress</h2>
          <p className="text-slate-500">Processing job <span className="font-mono font-bold text-slate-900">#{jobId}</span></p>
        </div>

        <div className="w-full space-y-6">
          <div className="space-y-4">
            {stages.map((stage, i) => (
              <div key={i} className={`flex items-start space-x-4 p-4 rounded-2xl border transition-all duration-300 ${
                i === progressStage ? 'bg-white border-indigo-200 shadow-md scale-105' : 
                i < progressStage ? 'bg-slate-50 border-slate-100 opacity-60' : 
                'bg-transparent border-transparent opacity-30 grayscale'
              }`}>
                <div className={`p-2 rounded-lg ${i <= progressStage ? 'bg-indigo-100 text-indigo-600' : 'bg-slate-100 text-slate-400'}`}>
                  {i < progressStage ? <CheckCircle size={16} /> : stage.icon}
                </div>
                <div className="flex-1">
                  <p className={`text-sm font-bold ${i <= progressStage ? 'text-slate-900' : 'text-slate-400'}`}>{stage.label}</p>
                  {i === progressStage && <p className="text-xs text-slate-500 mt-0.5">{stage.detail}</p>}
                </div>
                {i === progressStage && <RefreshCw size={14} className="animate-spin text-indigo-400 mt-1" />}
              </div>
            ))}
          </div>

          <div className="w-full h-2 bg-slate-100 rounded-full overflow-hidden">
            <div 
              className="h-full bg-indigo-600 transition-all duration-700 ease-out"
              style={{ width: `${((progressStage + 1) / stages.length) * 100}%` }}
            />
          </div>
        </div>
        
        <p className="text-xs text-slate-400 italic">No real emails are sent during this process.</p>
      </div>
    );
  }

  return (
    <div className="space-y-8 animate-in fade-in duration-500">
      <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
        <div>
          <div className="flex items-center space-x-2 text-sm text-slate-400 mb-1">
            <span onClick={() => navigate('/dashboard')} className="hover:text-slate-600 cursor-pointer">Dashboard</span>
            <ChevronRight size={14} />
            <span onClick={() => navigate('/history')} className="hover:text-slate-600 cursor-pointer">History</span>
            <ChevronRight size={14} />
            <span>Job {jobId}</span>
          </div>
          <h1 className="text-3xl font-bold text-slate-900">Verification Results</h1>
        </div>
        <div className="flex items-center space-x-3">
          <a href={verificationService.exportTxtUrl(jobId!)} className="px-4 py-2.5 bg-white border border-slate-200 rounded-xl text-slate-700 font-semibold text-sm hover:bg-slate-50 transition-colors flex items-center space-x-2">
            <Download size={18} />
            <span>Export TXT</span>
          </a>
          <a href={verificationService.exportCsvUrl(jobId!)} className="px-4 py-2.5 bg-indigo-600 text-white rounded-xl font-semibold text-sm hover:bg-indigo-700 transition-colors shadow-lg shadow-indigo-200 flex items-center space-x-2">
            <Download size={18} />
            <span>Export CSV</span>
          </a>
        </div>
      </div>

      <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
        <div className="bg-white p-5 rounded-2xl border border-slate-200 shadow-sm">
          <p className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-1">Valid</p>
          <div className="flex items-end justify-between">
            <p className="text-2xl font-bold text-slate-900">{stats[VerificationStatus.VALID] || 0}</p>
            <div className="text-emerald-500 bg-emerald-50 px-2 py-0.5 rounded-md text-xs font-bold">Safe</div>
          </div>
        </div>
        <div className="bg-white p-5 rounded-2xl border border-slate-200 shadow-sm">
          <p className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-1">Invalid</p>
          <div className="flex items-end justify-between">
            <p className="text-2xl font-bold text-slate-900">{stats[VerificationStatus.INVALID] || 0}</p>
            <div className="text-rose-500 bg-rose-50 px-2 py-0.5 rounded-md text-xs font-bold">Hard Bounce</div>
          </div>
        </div>
        <div className="bg-white p-5 rounded-2xl border border-slate-200 shadow-sm">
          <p className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-1">Catch-All</p>
          <div className="flex items-end justify-between">
            <p className="text-2xl font-bold text-slate-900">{stats[VerificationStatus.CATCH_ALL] || 0}</p>
            <div className="text-amber-500 bg-amber-50 px-2 py-0.5 rounded-md text-xs font-bold">Risky</div>
          </div>
        </div>
        <div className="bg-white p-5 rounded-2xl border border-slate-200 shadow-sm">
          <p className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-1">Unknown</p>
          <div className="flex items-end justify-between">
            <p className="text-2xl font-bold text-slate-900">{stats[VerificationStatus.UNKNOWN] || 0}</p>
            <div className="text-slate-500 bg-slate-100 px-2 py-0.5 rounded-md text-xs font-bold">Timed Out</div>
          </div>
        </div>
      </div>

      <div className="bg-white rounded-2xl border border-slate-200 shadow-sm overflow-hidden">
        <div className="p-4 bg-slate-50 border-b border-slate-100 flex flex-col md:flex-row md:items-center justify-between gap-4">
          <div className="flex items-center space-x-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-slate-400" size={16} />
              <input 
                type="text" 
                placeholder="Filter results..." 
                className="pl-9 pr-4 py-2 border border-slate-200 rounded-lg text-sm focus:ring-2 focus:ring-indigo-500 outline-none w-64"
                value={search}
                onChange={(e) => setSearch(e.target.value)}
              />
            </div>
            <select 
              className="bg-white border border-slate-200 rounded-lg py-2 px-3 text-sm focus:ring-2 focus:ring-indigo-500 outline-none"
              value={filter}
              onChange={(e) => setFilter(e.target.value as any)}
            >
              <option value="All">All Statuses</option>
              <option value={VerificationStatus.VALID}>Valid Only</option>
              <option value={VerificationStatus.INVALID}>Invalid Only</option>
              <option value={VerificationStatus.CATCH_ALL}>Risky Only</option>
            </select>
          </div>
          <div className="text-xs text-slate-400 flex items-center bg-white px-3 py-1.5 rounded-full border border-slate-100">
            <AlertCircle size={14} className="mr-1.5" />
            No real emails are sent during verification.
          </div>
        </div>

        <div className="overflow-x-auto">
          <table className="w-full">
            <thead>
              <tr className="bg-slate-50 text-left">
                <th className="px-6 py-4 text-xs font-bold text-slate-500 uppercase tracking-widest">Email Address</th>
                <th className="px-6 py-4 text-xs font-bold text-slate-500 uppercase tracking-widest text-center">Status</th>
                <th className="px-6 py-4 text-xs font-bold text-slate-500 uppercase tracking-widest text-center">AI Confidence</th>
                <th className="px-6 py-4 text-xs font-bold text-slate-500 uppercase tracking-widest">Technical Reason</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-slate-100">
              {filteredResults?.map((res, i) => (
                <tr key={i} className="hover:bg-slate-50 transition-colors">
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-semibold text-slate-900">{res.email}</td>
                  <td className="px-6 py-4 whitespace-nowrap text-center">
                    {getStatusBadge(res.status)}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-center">
                    <div className="flex flex-col items-center group relative">
                      <div className="w-24 h-2 bg-slate-100 rounded-full overflow-hidden">
                        <div 
                          className={`h-full rounded-full ${
                            res.confidence > 90 ? 'bg-emerald-500' : 
                            res.confidence > 70 ? 'bg-amber-500' : 
                            'bg-rose-500'
                          }`}
                          style={{ width: `${res.confidence}%` }}
                        />
                      </div>
                      <span className="text-xs font-bold text-slate-500 mt-1">{res.confidence}%</span>
                      
                      <div className="absolute bottom-full mb-2 hidden group-hover:block w-48 p-2 bg-slate-900 text-white text-[10px] rounded-lg shadow-xl z-20">
                        Confidence reflects SMTP acceptance and domain behavior. Not a guarantee of inbox delivery.
                      </div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-slate-500">
                    <div className="flex items-center space-x-2">
                      <Shield size={14} className="text-slate-300" />
                      <span>{res.reason}</span>
                    </div>
                  </td>
                </tr>
              ))}
              {(!filteredResults || filteredResults.length === 0) && (
                <tr>
                  <td colSpan={4} className="px-6 py-12 text-center text-slate-400">
                    <Search size={32} className="mx-auto mb-3 opacity-20" />
                    No results matching your filters.
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );
};

export default Results;
