
import React, { useState, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import { Upload, Mail, AlertCircle, X, CheckCircle, RefreshCw, ShieldCheck } from 'lucide-react';
import { verificationService } from '../services/api';
import { User } from '../types';

interface VerifyProps { user: User | null; }

const Verify: React.FC<VerifyProps> = ({ user }) => {
  const navigate = useNavigate();
  const [input, setInput] = useState('');
  const [emails, setEmails] = useState<string[]>([]);
  const [isProcessing, setIsProcessing] = useState(false);
  const [isDragging, setIsDragging] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const parseInput = (text: string) => {
    const list = text.split(/[\n,\s\t]+/)
      .map(e => e.trim().toLowerCase())
      .filter(e => e.length > 0);
    
    const unique = Array.from(new Set(list));
    setEmails(unique);
    setInput(text);
  };

  const processFile = (file: File) => {
    if (!file) return;
    const reader = new FileReader();
    reader.onload = (event) => {
      const text = event.target?.result as string;
      parseInput(text);
    };
    reader.readAsText(file);
  };

  const handleFileUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) processFile(file);
  };

  const handleDragOver = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(true);
  }, []);

  const handleDragLeave = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
  }, []);

  const handleDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
    
    const file = e.dataTransfer.files?.[0];
    if (file) {
      processFile(file);
    }
  }, []);

  const handleSubmit = async () => {
    if (emails.length === 0) {
      setError('Please provide at least one email address.');
      return;
    }

    const emailRegex = /^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;
    const invalidEmails = emails.filter(email => !emailRegex.test(email));

    if (invalidEmails.length > 0) {
      const displayInvalid = invalidEmails.slice(0, 2).join(', ');
      const suffix = invalidEmails.length > 2 ? '...' : '';
      setError(`Found ${invalidEmails.length} invalid email format(s): ${displayInvalid}${suffix}. Please correct them before proceeding.`);
      return;
    }

    if ((user?.credits || 0) < emails.length) {
      setError('Insufficient credits for this verification job.');
      return;
    }

    setIsProcessing(true);
    setError(null);

    try {
      const { jobId } = await verificationService.verifyBulk(emails);
      navigate(`/results/${jobId}`);
    } catch (err: any) {
      setError(err.response?.data?.message || 'An error occurred while starting the verification job.');
      setIsProcessing(false);
    }
  };

  return (
    <div className="max-w-4xl mx-auto space-y-8 animate-in slide-in-from-bottom-4 duration-500">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-slate-900">Verify Leads</h1>
          <p className="text-slate-500 mt-1">Paste your list or upload a CSV file to start.</p>
        </div>
        <div className="bg-white border border-slate-200 px-4 py-2 rounded-xl flex items-center space-x-2 text-sm font-medium">
          <span className="text-slate-500">Available Credits:</span>
          <span className="text-indigo-600 font-bold">{user?.credits?.toLocaleString() || 0}</span>
        </div>
      </div>

      <div className="bg-white rounded-2xl border border-slate-200 shadow-xl overflow-hidden">
        <div className="p-8">
          <div className="space-y-6">
            <div>
              <label className="block text-sm font-bold text-slate-700 mb-2">Paste Emails</label>
              <textarea 
                className={`w-full h-64 p-4 bg-slate-50 border rounded-xl focus:ring-2 focus:ring-indigo-500 focus:bg-white transition-all resize-none font-mono text-sm ${error && error.includes('invalid') ? 'border-red-300 ring-red-100 ring-2' : 'border-slate-200'}`}
                placeholder="Enter emails (one per line, comma separated, or paste from sheet)..."
                value={input}
                onChange={(e) => parseInput(e.target.value)}
              />
            </div>

            <div className="relative">
              <div className="absolute inset-0 flex items-center" aria-hidden="true">
                <div className="w-full border-t border-slate-200"></div>
              </div>
              <div className="relative flex justify-center text-sm font-medium">
                <span className="px-4 bg-white text-slate-500">OR</span>
              </div>
            </div>

            <div className="flex items-center justify-center w-full">
              <label 
                onDragOver={handleDragOver}
                onDragLeave={handleDragLeave}
                onDrop={handleDrop}
                className={`flex flex-col items-center justify-center w-full h-32 border-2 border-dashed rounded-xl cursor-pointer transition-all ${
                  isDragging 
                    ? 'border-indigo-500 bg-indigo-50 scale-[1.01] shadow-inner' 
                    : 'border-slate-300 bg-slate-50 hover:bg-slate-100'
                }`}
              >
                <div className="flex flex-col items-center justify-center pt-5 pb-6">
                  <Upload className={`w-8 h-8 mb-3 transition-colors ${isDragging ? 'text-indigo-500' : 'text-slate-400'}`} />
                  <p className="mb-2 text-sm text-slate-600">
                    <span className="font-bold">Click to upload</span> or drag and drop
                  </p>
                  <p className="text-xs text-slate-500">CSV, TXT, or Excel (MAX. 50MB)</p>
                </div>
                <input type="file" className="hidden" accept=".csv,.txt" onChange={handleFileUpload} />
              </label>
            </div>
          </div>
        </div>

        <div className="bg-slate-50 p-8 border-t border-slate-200 flex flex-col md:flex-row items-center justify-between space-y-4 md:space-y-0">
          <div className="flex items-center space-x-8">
            <div className="text-center">
              <p className="text-xs text-slate-500 uppercase font-bold tracking-wider">Detected</p>
              <p className="text-2xl font-bold text-slate-900">{emails.length}</p>
            </div>
            <div className="h-10 w-px bg-slate-200" />
            <div className="text-center">
              <p className="text-xs text-slate-500 uppercase font-bold tracking-wider">Duplicates</p>
              <p className="text-2xl font-bold text-amber-600">0</p>
            </div>
            <div className="h-10 w-px bg-slate-200" />
            <div className="text-center">
              <p className="text-xs text-slate-500 uppercase font-bold tracking-wider">Required</p>
              <p className="text-2xl font-bold text-indigo-600">{emails.length} <span className="text-sm font-medium">Credits</span></p>
            </div>
          </div>

          <div className="flex flex-col items-end space-y-4">
            {error && (
              <div className="flex items-center space-x-2 text-red-500 text-sm font-medium bg-red-50 px-3 py-2 rounded-lg border border-red-100 max-w-md">
                <AlertCircle size={16} className="shrink-0" />
                <span>{error}</span>
              </div>
            )}
            <button 
              onClick={handleSubmit}
              disabled={isProcessing || emails.length === 0}
              className={`px-10 py-4 bg-indigo-600 text-white rounded-xl font-bold text-lg shadow-lg shadow-indigo-200 flex items-center space-x-3 transition-all ${
                isProcessing || emails.length === 0 ? 'opacity-50 cursor-not-allowed grayscale' : 'hover:bg-indigo-700 hover:-translate-y-0.5'
              }`}
            >
              {isProcessing ? (
                <>
                  <RefreshCw className="animate-spin" size={20} />
                  <span>Processing...</span>
                </>
              ) : (
                <>
                  <ShieldCheck size={20} />
                  <span>Verify Leads</span>
                </>
              )}
            </button>
          </div>
        </div>
      </div>

      <div className="flex items-center justify-center space-x-12 pt-8 text-slate-400">
        <div className="flex items-center space-x-2 grayscale opacity-50">
          <CheckCircle size={18} />
          <span className="text-sm font-medium">Safe Handshake</span>
        </div>
        <div className="flex items-center space-x-2 grayscale opacity-50">
          <CheckCircle size={18} />
          <span className="text-sm font-medium">GDPR Compliant</span>
        </div>
        <div className="flex items-center space-x-2 grayscale opacity-50">
          <CheckCircle size={18} />
          <span className="text-sm font-medium">No Spamming</span>
        </div>
      </div>

      <div className="bg-white/60 p-6 rounded-2xl border border-dashed border-slate-300 text-center">
        <p className="text-slate-500 text-sm italic">
          "No real emails are sent during the verification process. We perform low-level SMTP handshakes to verify inbox existence without disturbing the recipient."
        </p>
      </div>
    </div>
  );
};

export default Verify;
