
import axios from 'axios';
import { User, Job, VerificationResult } from '../types';

// Production backend is hosted at the /v1/ subdirectory of mailvity.cc
const api = axios.create({
  baseURL: 'https://mailvity.cc/v1',
  headers: {
    'Content-Type': 'application/json',
  },
});

// Interceptor to attach the JWT token from localStorage to every outgoing request
api.interceptors.request.use((config) => {
  const token = localStorage.getItem('token');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

export const authService = {
  login: async (credentials: any) => {
    const response = await api.post('/auth/login.php', credentials);
    return response.data;
  },
  register: async (data: any) => {
    const response = await api.post('/auth/register.php', data);
    return response.data;
  },
};

export const billingService = {
  initiatePayment: async (planId: string, method: 'stripe' | 'crypto' = 'stripe') => {
    const response = await api.post('/payment_init.php', { planId, method });
    return response.data; // Expected { url: '...' }
  }
};

export const verificationService = {
  getCredits: async () => {
    const response = await api.get('/credits.php');
    return response.data.credits;
  },
  verifyBulk: async (emails: string[]) => {
    const response = await api.post('/verify.php', { emails });
    return response.data;
  },
  getResults: async (jobId: string): Promise<Job> => {
    const response = await api.get(`/results.php?jobId=${jobId}`);
    return response.data;
  },
  getHistory: async () => {
    const response = await api.get('/history.php');
    return response.data;
  },
  exportCsvUrl: (jobId: string) => `https://mailvity.cc/v1/export.php?type=csv&jobId=${jobId}&token=${localStorage.getItem('token')}`,
  exportTxtUrl: (jobId: string) => `https://mailvity.cc/v1/export.php?type=txt&jobId=${jobId}&token=${localStorage.getItem('token')}`,
};

export const adminService = {
  getUsers: async () => {
    const response = await api.get('/admin/users.php');
    return response.data;
  },
  updateUser: async (userId: string, data: Partial<User>) => {
    const response = await api.post('/admin/users.php?action=update', { userId, ...data });
    return response.data;
  },
  deleteUser: async (userId: string) => {
    const response = await api.post('/admin/users.php?action=delete', { userId });
    return response.data;
  },
  getSystemStats: async () => {
    const response = await api.get('/admin/stats.php');
    return response.data;
  }
};

export default api;
